#!/usr/bin/env python

# Export to SMF forum gallery (gimpuj.info) functionality
# and localisation support.
# Copyright (C) 2009 Siraiye and Ziomioslaw
# http://www.gimpuj.info
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

import urllib2, cookielib, os, sys, urllib, mimetools, mimetypes, re, libpub
from cStringIO import StringIO
from htmlentitydefs import name2codepoint
from libpub.lang import __

class GimpujObject:
    """
    class GimpujObject
    Main class providing basic functions to manage connection with gimpuj.info.
    """
    
    COOKIEFILE = '.cookies.lwp'
    # default browser to fake
    DEFBROWSER = 'Mozilla/5.0 (Windows; U; Windows NT 5.1; pl; rv:1.9.0.3)'

    def __init__(self):

        # set initial values to variables
        self.load_configuration()
        
        # create a cookie jar
        self.cj = cookielib.LWPCookieJar()

        if os.path.isfile(self.COOKIEFILE):
            # if we have a cookie file already saved load the cookies into the Cookie Jar
            self.cj.load(self.COOKIEFILE)

        # Now we need to get our Cookie Jar installed in the opener for fetching URLs
        self.opener = urllib2.build_opener(urllib2.HTTPCookieProcessor(self.cj))
        urllib2.install_opener(self.opener)
        
        # get last used username and password if they exist
        last_username = libpub.conf.get('GIMPUJ_LAST_USERNAME')
        last_password_cipher = libpub.conf.get('GIMPUJ_LAST_PASSWORD')
        if last_password_cipher:
            last_password = libpub.utils.decrypt(last_password_cipher)
        else:
            last_password = None
            
        # and login using saved values
        if last_username and last_password:
            try:
                self.login(last_username, last_password)
            except:
                pass
        
        
    def load_configuration(self):
        """
        load_configuration
        Method setting initial values.
        """
        
        # get URLs from config file
        self.LOGINURL = libpub.conf.get('LOGINURL')
        self.LOGOUTURL = libpub.conf.get('LOGOUTURL')
        self.ADDPICURL = libpub.conf.get('ADDPICURL')
        self.ADDURL = libpub.conf.get('ADDURL')
        self.UIDURL = libpub.conf.get('UIDURL')
        
        self.categories = {}    # albums in gallery
        self.uid = ''           # user id
        self._has_auth = False  # did login succeed?
        
        
    def create_header(self):
        """
        create_header
        Helper method creating default header with cookies for POST request.
        """
        
        # fake a user agent, some websites (like google) don't like automated exploration
        headers =  {'User-agent' : self.DEFBROWSER}
        cookiesToSet = ""
        # get all the cookies from cookie jar
        for index, cookie in enumerate(self.cj):
            cookiesToSet += cookie.name + '=' + cookie.value + ';'
        # and set them in the header
        headers['Cookie'] = cookiesToSet
        
        return headers
    
    
    def get_content_type(self, filename):
        """
        get_content_type
        Helper method returning file mimetype.
        """
        return mimetypes.guess_type(filename)[0] or 'application/octet-stream'

    
    def encode_multipart_formdata(self, fields, files):
        """
        encode_multipart_formdata
        Method creating body for multipart/form-data POST request.
        fields is a sequence of (name, value) elements for regular form fields.
        files is a sequence of (name, filename, value) elements for data to be uploaded as files
        Returns (content_type, body) ready for httplib.HTTP instance
        """
        # create a boundary
        boundary = mimetools.choose_boundary()
        # create a buffer for body
        buf = StringIO()
        # add values of all regular fields
        for (key, value) in fields:
            buf.write('--%s\r\n' % boundary)
            buf.write('Content-Disposition: form-data; name="%s"' % key)
            buf.write('\r\n\r\n' + value + '\r\n')
        # add files from all file fields
        for (key, filename, value) in files:
            buf.write('--%s\r\n' % boundary)
            buf.write('Content-Disposition: form-data; name="%s"; filename="%s"\r\n' % (key, filename))
            buf.write('Content-Type: %s\r\n' % self.get_content_type(filename))
            buf.write('Content-Transfer-Encoding: binary\r\n')
            value.seek(0)
            buf.write('\r\n' + value.read() + '\r\n')
        buf.write('--' + boundary + '--')
        
        body = buf.getvalue()
        content_type = 'multipart/form-data; boundary=%s' % boundary
        
        return content_type, body


    def send_request(self, theurl, txdata, txheaders):
        """
        send_request
        Method sending a POST request.
        theurl - url to send request to,
        txdata - body of the request,
        txheaders - request headers.
        Returns handle to request result.
        """
        try:
            # create a request object
            req = urllib2.Request(theurl, txdata, txheaders)
            # and open it to return a handle on the url
            handle = urllib2.urlopen(req)
            
        except IOError, e:
            errStr = 'We failed to open "%s"' % theurl
            if hasattr(e, 'code'):
                errStr += ' with error code %s' % e.code
            elif hasattr(e, 'reason'):
                errStr += ". Reason: " + str(e.reason)
            raise GimpujException, errStr
            
        else:
            return handle

        
    def login(self, user, password):
        """
        login
        Method performing user login.
        Sets self._has_auth to True if succeeded, otherwise sets it to False.
        Returns result page in html.
        """

        # fake a user agent, some websites (like google) don't like automated exploration
        headers =  {'User-agent' : self.DEFBROWSER}
        # create body of the request
        body = {'user': user.encode('iso-8859-2'), 'passwrd': password.encode('iso-8859-2'), 'cookielength' : '60'}
        # send the login request
        resp = self.send_request(self.LOGINURL, urllib.urlencode(body), headers)
        # save received cookies
        self.cj.save(self.COOKIEFILE)
        # get the result page
        page = resp.read()
        # check if login succeeded
        if resp.geturl() != self.LOGINURL:
            # if so, set self._has_auth
            self._has_auth = True
            # get user id based on request result
            self.uid = self.get_uid(page)
            # get all available albums
            self.categories = self.get_categories()
        else:
            # otherwise, reset self._has_auth and throw the exception
            self._has_auth = False
            # empty categories
            self.categories = {}
            raise GimpujException, __("Wrong username or password.")
        
        return page
    
    
    def logout(self):
        """
        logout
        Cleanup method.
        """
        # empty cookie jar
        self.cj.clear()
        self.cj.save(self.COOKIEFILE) 


    def send_picture(self, filename, category, title, description = '', keywords = ''):
        """
        send_picture
        Method sending picture described by filename, category, title, description and keywords.
        Returns True if operation succeeded, throws GimpujException otherwise.
        """

        headers = self.create_header()

        # get separator specific for every system
        if sys.platform.find('win32') >= 0:
            index = filename.rfind('\\')
        else:
            index = filename.rfind('/')
        # get only the name of the file    
        name = filename[(index + 1):]
            
        # create the request body and headers
        fields = (('title', title.encode('iso-8859-2')), ('cat', category), ('description', description.encode('iso-8859-2')), ('keywords', keywords.encode('iso-8859-2')), ('userid', self.uid), ('allowcomments', 'checked'), ('sendemail', 'checked'))
        files = (('picture', name, open(filename, 'rb')),)
        content_type, body = self.encode_multipart_formdata(fields, files)
        headers['Content-Type'] = content_type
        
        # send the request
        resp = self.send_request(self.ADDPICURL, body, headers)
        
        # check if operation succeeded
        if resp.geturl() != self.ADDPICURL:
            return True
        # otherwise throw the exception
        else:
            raise GimpujException, __('Upload failed.')

            
    def has_auth(self):
        """
        has_auth
        Getter method.
        """
        return self._has_auth


    def htmlentitydecode(self, s):
        """
        htmlentitydecode
        Method decoding all html entities in a string,
        i.e. converts &gt; into >
        """
        return re.sub('&(%s);' % '|'.join(name2codepoint), 
            lambda m: unichr(name2codepoint[m.group(1)]), s)


    def get_categories(self):
        """
        get_categories
        Method checking for available albums.
        """
        # get body of the page
        headers = self.create_header()
        req = urllib2.Request(self.ADDURL + self.uid, headers=headers)
        body = urllib2.urlopen(req).read()
        self.cj.save(self.COOKIEFILE) 
        
        # extract all the album names and return the dictionary in following format:
        # "album_name" : "album_number"
        m = re.findall('<option value="([0-9]+)"[^>]*>([^<]+)</option>', body)
        # get every album except (none) and decode its name
        m = [(unicode(self.htmlentitydecode(tup[1]), 'iso-8859-2'), tup[0]) for tup in m if tup[0] != '0']
        
        return dict(m)
        
        
    def get_uid(self, page):
        """
        get_uid
        Method extracting user id from the page returned after login.
        """
        m = re.search(self.UIDURL + '([0-9]+)', page)
        if m != None:
            return m.group(1)
        else:
            return ''


class GimpujException(Exception):
    """
    class GimpujException
    Gimpuj own exception type.
    """
    
    def __init__(self, value):
        self.value = value
    
    def __str__(self):
        return str(self.value)

